library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;
use std.textio.all;

entity sqrt_tb is
end sqrt_tb;

architecture Test of sqrt_tb is

    -- Podesi ove konstante prema svom projektu
    constant C_IN_BW    : integer := 16; -- Broj bita ulaza
    constant C_OUT_BW   : integer := 16; -- Broj bita izlaza
    constant C_OUT_FRAC : integer := 8; -- Broj bita za razlomljeni deo izlaza
    constant C_CLK_PERIOD : time := 10 ns;

    -- Signali za interfejs
    signal clk       : std_logic := '1';
    signal reset     : std_logic := '0';
    signal d_in      : std_logic_vector(C_IN_BW-1 downto 0) := (others => '0');
    signal valid_in  : std_logic := '0';
    signal d_out     : std_logic_vector(C_OUT_BW-1 downto 0);
    signal valid_out : std_logic := '0';
    -- Ocekivana vrednost procitana iz fajla
    signal d_out_exp : std_logic_vector(C_OUT_BW-1 downto 0);
    
begin

    -- Generisanje takta
    clk <= not clk after C_CLK_PERIOD/2;

    ---------------------------------------------------------------------------
    -- DUMMY MODEL: Simulira modul korena - potrebno ga je zameniti instancom
    -- realiyovane hardverske komponente
    ---------------------------------------------------------------------------
    dummy_logic: process
        variable v_input_val  : real;
        variable v_root_val   : real;
        variable v_scaled_res : integer;
    begin
        if reset = '1' then
            valid_out <= '0';
            d_out     <= (others => '0');
            wait for C_CLK_PERIOD;
        else
            if valid_in = '1' then
                -- Konverzija ulaznog logic vektora u integer, pa u realni broj
                v_input_val := real(to_integer(unsigned(d_in)));
                -- Racunanje korena
                v_root_val := sqrt(v_input_val);
                -- Skaliranje za fiksni zarez (mnozenje sa 2^C_OUT_FRAC) da bi se dobila fixed point predstava
                v_scaled_res := integer(trunc(v_root_val * real(2**C_OUT_FRAC)));
                
                -- Simuliraj kanjenje obrade (npr. 5 taktova)
                wait for 5*C_CLK_PERIOD;
                
                -- Postavljanje izlaza i valid_out signala na jedan takt
                d_out <= std_logic_vector(to_unsigned(v_scaled_res, C_OUT_BW));
                valid_out <= '1';
                wait for C_CLK_PERIOD;
                valid_out <= '0';
            else
                -- Ako ulaz nije validan, cekamo naredni takt
                wait for C_CLK_PERIOD;
                valid_out <= '0';
            end if;
        end if;
    end process;

    ---------------------------------------------------------------------------
    -- TEST PROCES: Cita fajlove i proverava rezultate
    ---------------------------------------------------------------------------
    stim_proc: process
        file in_file  : text;
        file ref_file : text;
        variable v_in_line, v_ref_line : line;
        variable v_in_vec  : bit_vector(C_IN_BW-1 downto 0);
        variable v_ref_vec : bit_vector(C_OUT_BW-1 downto 0);
    begin
        -- ne zaboravite da dodate fajlove kao Simulation Sources
        file_open(in_file, "sqrt_input.txt", read_mode);
        file_open(ref_file, "sqrt_output.txt", read_mode);
        -- Inicijalni reset
        reset <= '1';
        wait for C_CLK_PERIOD*2;
        reset <= '0';
        wait for C_CLK_PERIOD;

        while not endfile(in_file) loop
            -- Citanje iz fajlova
            readline(in_file, v_in_line);
            read(v_in_line, v_in_vec);
            
            readline(ref_file, v_ref_line);
            read(v_ref_line, v_ref_vec);

            -- Slanje podataka dummy modelu
            d_in     <= to_stdlogicvector(v_in_vec);
            valid_in <= '1';
            wait for C_CLK_PERIOD; -- moze i duze, prema tekstu zadatka
            valid_in <= '0';

            -- Cekanje na odgovor
            wait until valid_out = '1';
            
            -- konverzija ocekivane vrednosti
            d_out_exp <= to_stdlogicvector(v_ref_vec);
            -- Provera
            wait for 0 ns; -- forsiranje azuriranja signala
            assert (d_out = d_out_exp) -- desice se ako su rezultati razliciti i ispisace se greska u konzoli
                report "GRESKA: Dobijeno " & to_hstring(to_bitvector(d_out)) & 
                       ", ocekivano " & to_hstring(v_ref_vec)
                severity error;
            
            wait for C_CLK_PERIOD;
        end loop;
        wait;
    end process;
    
    ---------------------------------------------------------------------------
    -- PRIMER UPISA U FAJL: Nije relevantno za ovaj test korena, ali moze da 
    -- koristi. Neophodan je upis u fajl za proveru rezultata obrade slike, pa
    -- se ovaj proces moze koristiti kao polazna osnova. Ovaj primer na svaki
    -- takt upisuje po jednu od 100 random integer vrednosti u fajl i to kao 
    -- niz bita.
    ---------------------------------------------------------------------------
    file_write_proc: process
        file out_file   : text;
        variable v_out_line : line;
        variable v_out_vec  : bit_vector(C_IN_BW-1 downto 0);
       
        variable seed1          : positive := 123;
        variable seed2          : positive := 456;
        variable v_rand_val     : real;
        variable v_rand_int_val : integer;
    begin
        -- Otvoriti fajl za upis. Ako se ne zada apsolutna putanja, lokacija fajla 
        -- ce biti project_dir.sim/sim_1/behav/xsim, gde je project_dir ime projekta.
        file_open(out_file, "test_out.txt", write_mode);
        
        for i in 1 to 100 loop
            -- Generisanje slucajnog broja (0.0 do 1.0)
            uniform(seed1, seed2, v_rand_val);
            -- Skaliranje na 16 bitni integer
            v_rand_int_val := integer(floor(v_rand_val * real(2**16)));
            -- Konverzija u std_logic_vector pa u bitvector
            v_out_vec := to_bitvector(std_logic_vector(to_unsigned(v_rand_int_val, 16)));
            -- Upis u fajl
            write(v_out_line, v_out_vec); -- Upisuje binarni string u liniju
            writeline(out_file, v_out_line); -- Upisuje liniju u fajl
            wait for C_CLK_PERIOD;
        end loop;
    
        file_close(out_file);
        wait;
    end process;

end Test;
